<?php

/**
 * Exports Events and registrations to a JSON File under Import/Export tab on settings page
 *
 * This file is used to markup the admin-facing aspects of the plugin.
 *
 * @link       //wpminds.com
 * @since      1.4.2
 *
 * @package    Wp_Events
 * @subpackage Wp_Events/admin/includes
 */

 function wp_get_ajax_events() {
	 //Verify Nonce
	 $security = isset( $_POST['security'] ) ? sanitize_text_field( wp_unslash( $_POST['security'] ) ) : '';
	 if ( ! wp_verify_nonce( $security, 'ajax-nonce' ) ) {
		 wp_send_json_error( 'Invalid nonce' );
	 } 
	global $wp_filesystem;

	if ( ! function_exists( 'request_filesystem_credentials' ) ) {
		require_once ABSPATH . 'wp-admin/includes/file.php';
	}

	$creds = request_filesystem_credentials( site_url() );
	if ( ! WP_Filesystem( $creds ) ) {
		wpe_send_ajax_response( array( 'error' => 'Filesystem access denied.' ) );
		return;
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'Unauthorized access', 'simple-wp-events' ) );
	}

	$args = array(
		'post_type'		 => 'wp_events',
		'posts_per_page' => -1,
		'post_status' 	 => array( 'publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash' ),
	);

	$event_status = isset( $_POST['postStatus'] ) ? sanitize_text_field( wp_unslash( $_POST['postStatus'] ) ) : '';

	if ( $event_status === 'Past' ) {
		$args['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
			[
				'key'     => 'wpevent-end-date-time',
				'compare' => '<',
				'value'   => strtotime( current_time( 'mysql' ) ),
				'type'    => 'numeric',
			],
		];
	} else if ( $event_status === 'Future' ) {
		$args['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
			[
				'key'     => 'wpevent-start-date-time',
				'compare' => '>',
				'value'   => strtotime( current_time( 'mysql' ) ),
				'type'    => 'numeric',
			],
		];
	} else if ( $event_status === 'On Going' ) {
		$args['meta_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
			'relation' => 'AND',
			[
				'key'     => 'wpevent-start-date-time',
				'compare' => '<=',
				'value'   => strtotime( current_time( 'mysql' ) ),
				'type'    => 'numeric',
			],
			[
				'key'     => 'wpevent-end-date-time',
				'compare' => '>=',
				'value'   => strtotime( current_time( 'mysql' ) ),
				'type'    => 'numeric',
			],
		];
	}

	$events = new WP_Query( $args );

	$path = wp_upload_dir();
	$filename = '/events.csv';
	$full_path = $path['path'] . $filename;
	$file_content = '';

	$event_content = array();

	if ( $events->have_posts() ) {
		while ( $events->have_posts() ) : $events->the_post();
			$post_id         = get_the_ID();
			$event_name      = get_the_title( $post_id );
			$event_date_time = wpevent_date_time( $post_id );
			$start_date      = $event_date_time['start_date'];
			$start_time      = $event_date_time['start_time'];
			$end_date        = $event_date_time['end_date'];
			$end_time        = $event_date_time['end_time'];
			$end_date_time   = get_post_meta( $post_id, 'wpevent-end-date-time', TRUE );
			$wpe_venue       = get_post_meta( $post_id, 'wpevent-venue', TRUE );
			$wpe_addr        = get_post_meta( $post_id, 'wpevent-address', TRUE );
			$wpe_city        = get_post_meta( $post_id, 'wpevent-city', TRUE );
			$wpe_state       = get_post_meta( $post_id, 'wpevent-state', TRUE );
			$wpe_country     = get_post_meta( $post_id, 'wpevent-country', TRUE );
			$seats           = get_post_meta( $post_id, 'wpevent-seats', TRUE );
			$booked_seats    = get_booked_seats( $post_id );
			$gmap_url        = get_post_meta( $post_id, 'wpevent-map-url', TRUE );
			$wpe_type        = get_post_meta( $post_id, 'wpevent-type', TRUE );
			$wpe_phone       = get_post_meta( $post_id, 'wpevent-phone', TRUE );

			$event_content[] = array (
				'Post ID' 		 => $post_id,
				'Event Name'     => $event_name,
				'Start Date' 	 => $start_date,
				'End Date' 		 => $end_date,
				'Start Time' 	 => $start_time,
				'End Time' 		 => $end_time,
				'Venue' 		 => $wpe_venue,
				'Address' 		 => $wpe_addr,
				'City' 			 => $wpe_city,
				'State' 		 => $wpe_state,
				'Country' 		 => $wpe_country,
				'Total Seats' 	 => $seats,
				'Booked Seats' 	 => $booked_seats,
				'Google Map URL' => $gmap_url,
				'Events Type'	 => $wpe_type,
				'Event Status'   => get_post_status( $post_id ),
				'Phone'			 => $wpe_phone,
			);
		endwhile;

		$keys = array_keys( $event_content[0] );
		$file_content .= implode( ',', $keys ) . "\n";
		foreach ( $event_content as $event_info ) {
			$file_content .= implode( ',', array_map( 'sanitize_csv_field', $event_info ) ) . "\n";
		}
	} else {
		$empty_row = array(
			'Post ID' 		 => '',
			'Event Name'     => '',
			'Start Date' 	 => '',
			'End Date' 		 => '',
			'Start Time' 	 => '',
			'End Time' 		 => '',
			'Venue' 		 => '',
			'Address' 		 => '',
			'City' 			 => '',
			'State' 		 => '',
			'Country' 		 => '',
			'Total Seats' 	 => '',
			'Booked Seats' 	 => '',
			'Google Map URL' => '',
			'Events Type'	 => '',
			'Event Status'   => '',
			'Phone'			 => '',
		);
		$keys = array_keys( $empty_row );
		$file_content .= implode( ',', $keys ) . "\n";
	}

	$wp_filesystem->put_contents( $full_path, $file_content, FS_CHMOD_FILE );
	$file_url = $path['url'] . $filename;
	wpe_send_ajax_response( $file_url );
}

/**
 * Helper function to sanitize CSV values (handles commas, quotes, etc.)
 */
function sanitize_csv_field( $field ) {
	$field = str_replace( '"', '""', $field );
	return '"' . $field . '"';
}


add_action('wp_ajax_wp_get_ajax_events', 'wp_get_ajax_events');

/**
* Fetches Subscribers for WP Events from Database
*
* @since 1.0.446
*
* @return array 
*
*/
function get_wpe_subscribers() {

	global $wpdb;
	$table_name = $wpdb->prefix . 'events_subscribers';
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
	$results    = $wpdb->get_results( $wpdb->prepare('SELECT * FROM '. $table_name) );

	foreach ( $results as $result ) {

		$status = $result->wpe_status;

		switch( $status ) {
			case 0:
				$entry_status = 'Trash';
				break;
			case 1:
				$entry_status = 'Active';
				break;
			case -1:
				$entry_status = 'Deleted Permanently';
				break;
			default:
				$entry_status = 'Active';
		}

		$data[] = [
			'Id'                  => esc_attr( $result->id ),
			'First Name' 		  => esc_attr( $result->subscriber_firstname ),
			'Last Name'  		  => esc_attr( $result->subscriber_lastname ),
			'Email'     		  => esc_attr( $result->subscriber_email ),
			'Phone'     		  => esc_attr( $result->subscriber_phone ),
			'Texting Permission'  => esc_attr( $result->subscriber_texting_permission ),
			'Time'                => esc_attr( $result->time_generated ),
			'Status'              => $entry_status,
		];
	
	}

	return $data;
}

/**
 * Export Registrations
 */

add_action('wp_ajax_wpe_event_entries_export', 'wpe_event_entries_export');

function wpe_event_entries_export() {

 //Verify Nonce
 $security = isset( $_POST['security'] ) ? sanitize_text_field( wp_unslash( $_POST['security'] ) ) : '';
 if ( ! wp_verify_nonce( $security, 'ajax-nonce' ) ) {
	 wp_send_json_error( 'Invalid nonce' );
 } 

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'Unauthorized access', 'simple-wp-events' ) );
	}

	$event_startDate = isset( $_POST['Startdate'] ) ? sanitize_text_field( wp_unslash( $_POST['Startdate'] ) ) : '';
	$event_endDate   = isset( $_POST['Enddate'] ) ? sanitize_text_field( wp_unslash( $_POST['Enddate'] ) ) : '';
	$wpeevent        = isset( $_POST['wpeeventid'] ) ? sanitize_text_field( wp_unslash( $_POST['wpeeventid'] ) ) : '';
	
	$path            = wp_upload_dir();
	$t               = time();
	$file_name       = "events-entries-$t.csv";
	$file_path       = $path['path'] . '/' . $file_name;

	$args = array(
		'post_type'      => 'wp_events',
		'posts_per_page' => -1,
		'post_status'    => array( 'publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash' ),
	);

	if ( $wpeevent ) {
		$args['post__in'] = explode( ",", $wpeevent );
	}

	if ( $event_startDate !== '' && $event_endDate !== '' ) {
		$args['meta_query'] = array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
			'relation' => 'AND',
			array(
				'key'     => 'wpevent-end-date-time',
				'compare' => '<=',
				'value'   => strtotime( $event_endDate . ' 23:59:59' ),
				'type'    => 'numeric',
			),
			array(
				'key'     => 'wpevent-start-date-time',
				'compare' => '>=',
				'value'   => strtotime( $event_startDate ),
				'type'    => 'numeric',
			),
		);
	} elseif ( $event_startDate !== '' ) {
		$args['meta_query'] = array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
			array(
				'key'     => 'wpevent-start-date-time',
				'compare' => '>=',
				'value'   => strtotime( $event_startDate ),
				'type'    => 'numeric',
			),
		);
	} elseif ( $event_endDate !== '' ) {
		$args['meta_query'] = array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Used for filtering posts by meta fields, acceptable in admin context
			array(
				'key'     => 'wpevent-end-date-time',
				'compare' => '<=',
				'value'   => strtotime( $event_endDate . ' 23:59:59' ),
				'type'    => 'numeric',
			),
		);
	}

	$events = new WP_Query( $args );
	$data   = array();

	if ( $events->have_posts() ) {
		while ( $events->have_posts() ) {
			$events->the_post();
			$postID = get_the_ID();

			global $wpdb;
			$table_name = $wpdb->prefix . 'events_registration';
			$table_name = esc_sql( $table_name );
			$sql     = "SELECT * FROM {$table_name} WHERE post_id = %d";
			$results = $wpdb->get_results( $wpdb->prepare( $sql, $postID ) ); 	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared

			foreach ( $results as $result ) {
				switch ( $result->wpe_status ) {
					case 0:
						$status = 'Trash';
						break;
					case -1:
						$status = 'Deleted Permanently';
						break;
					case 1:
						$status = 'Active';
						break;
					case 2:
						$status = 'Pending';
						break;
					case 3:
						$status = 'Approved';
						break;
					case 4:
						$status = 'Cancelled';
						break;
					default:
						$status = 'Active';
				}

				$address1  = $result->addres_one;
				$address2  = ! empty( $result->addres_two ) ? ', ' . $result->addres_two . ', ' : ', ';
				$city      = $result->city;
				$state     = $result->state;
				$zip       = $result->zip;
				$full_addr = $address1 . $address2 . $city . ', ' . $state . ', ' . $zip;

				$data[] = array(
					'Id'                  => esc_attr( $result->ID ),
					'Event ID'            => esc_attr( $result->post_id ),
					'First Name'          => esc_attr( $result->first_name ),
					'Last Name'           => esc_attr( $result->last_name ),
					'Address'             => esc_attr( $full_addr ),
					'Email'               => esc_attr( $result->email ),
					'Phone'               => esc_attr( $result->phone ),
					'Event Name'          => get_the_title( $result->post_id ),
					'Event Type'          => get_post_meta( $result->post_id, 'wpevent-type', true ),
					'Time'                => esc_attr( $result->time_generated ),
					'Texting Permission'  => esc_attr( $result->texting_permission ),
					'Seats'               => esc_attr( $result->wpe_seats ),
					'Status'              => esc_attr( $status ),
				);
			}
		}
		wp_reset_postdata();
	}

	// Initialize WordPress filesystem
	global $wp_filesystem;
	require_once ABSPATH . 'wp-admin/includes/file.php';

	if ( ! WP_Filesystem() ) {
		wp_die( esc_html__( 'Unable to initialize WP Filesystem API.', 'simple-wp-events' ) );
	}

	// Build CSV content
	$file_content = '';
	if ( ! empty( $data ) ) {
		$headers = array_keys( $data[0] );
		$file_content .= implode( ',', $headers ) . "\n";
		foreach ( $data as $row ) {
			$file_content .= implode( ',', array_map( function( $value ) {
				$value = str_replace( '"', '""', $value ); 
				return '"' . $value . '"'; 
			}, $row ) ) . "\n";
		}
	} else {
		$headers = array(
			'Id',
			'Event ID',
			'First Name',
			'Last Name',
			'Address',
			'Email',
			'Phone',
			'Event Name',
			'Event Type',
			'Time',
			'Texting Permission',
			'Seats',
			'Status',
		);
		$file_content .= implode( ',', $headers ) . "\n";
	}

	// Write CSV file
	$wp_filesystem->put_contents(
		$file_path,
		$file_content,
		FS_CHMOD_FILE
	);

	// Send response with file URL
	$file_url = $path['url'] . '/' . $file_name;
	wpe_send_ajax_response( $file_url );
}

/**
 * 
 *  Export Subscriptions
 */
add_action('wp_ajax_wpe_export_subscription', 'wpe_export_subscription');

/**
 * Export Subscriptions to CSV file.
 * 
 * @since 1.4.3
 */
function wpe_export_subscription() {
	global $wp_filesystem;

	if ( empty( $wp_filesystem ) ) {
		require_once ABSPATH . '/wp-admin/includes/file.php';
		WP_Filesystem();
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'Unauthorized access', 'simple-wp-events' ) );
	}
	$subscribers = get_wpe_subscribers();
	$upload_dir  = wp_upload_dir();
	$t           = time();
	$filename    = "wpe-subscribers-{$t}.csv";
	$filepath    = trailingslashit( $upload_dir['path'] ) . $filename;

	// Prepare CSV content
	$csv_content = '';
	$keys        = array_keys( $subscribers[0] );
	$csv_content .= implode( ',', array_map( function( $value ) {
		$value = str_replace( '"', '""', $value );
		return '"' . $value . '"';
	}, $keys ) ) . "\n";

	foreach ( $subscribers as $entry ) {
		$csv_content .= implode( ',', array_map( function( $value ) {
			$value = str_replace( '"', '""', $value );
			return '"' . $value . '"';
		}, $entry ) ) . "\n";
	}

	// Write the file using WP_Filesystem
	$wp_filesystem->put_contents( $filepath, $csv_content, FS_CHMOD_FILE );

	// Return the URL to the generated file
	$file_url = trailingslashit( $upload_dir['url'] ) . $filename;
	wpe_send_ajax_response( $file_url );
}

/**
 * 
 *  Delete File
 */
add_action('wp_ajax_wpe_delete_file', 'wpe_delete_file');

/**
 * Delete CSV file after download.
 * 
 * @since 1.4.3
 */
function wpe_delete_file() {
	 //Verify Nonce
	 $security = isset( $_POST['security'] ) ? sanitize_text_field( wp_unslash( $_POST['security'] ) ) : '';
	 if ( ! wp_verify_nonce( $security, 'ajax-nonce' ) ) {
		 wp_send_json_error( 'Invalid nonce' );
	 } 
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'Unauthorized access', 'simple-wp-events' ) );
	}
	$file       = isset( $_POST['url'] ) ? sanitize_text_field( wp_unslash( $_POST['url'] ) ) : '';
	$upload_dir = wp_upload_dir();

	$relative_path = str_replace( $upload_dir['baseurl'], '', $file );
	$full_path     = $upload_dir['basedir'] . $relative_path;

	wp_delete_file( $full_path );

	wpe_send_ajax_response( 1 );
}